const { ipcRenderer } = require('electron');

class PixelArtApp {
    constructor() {
        // Canvas setup
        this.canvas = document.getElementById('pixelCanvas');
        this.ctx = this.canvas.getContext('2d', { willReadFrequently: true });
        this.overlayCanvas = document.getElementById('overlayCanvas');
        this.overlayCtx = this.overlayCanvas.getContext('2d');
        this.previewCanvas = document.getElementById('previewCanvas');
        this.previewCtx = this.previewCanvas.getContext('2d');

        // State
        this.gridWidth = 32;
        this.gridHeight = 32;
        this.pixelSize = 20;
        this.zoom = 1;
        this.showGrid = true;
        this.currentTool = 'pencil';
        this.primaryColor = '#000000';
        this.secondaryColor = '#ffffff';
        this.isDrawing = false;
        this.startPos = null;
        this.currentButton = 0;

        // Undo/Redo
        this.history = [];
        this.historyStep = -1;
        this.maxHistory = 50;

        // Default palette
        this.palette = [
            '#000000', '#FFFFFF', '#FF0000', '#00FF00', '#0000FF', '#FFFF00',
            '#FF00FF', '#00FFFF', '#FF8800', '#88FF00', '#0088FF', '#FF0088',
            '#808080', '#C0C0C0', '#800000', '#008000', '#000080', '#808000'
        ];

        this.init();
    }

    init() {
        this.initCanvas();
        this.setupEventListeners();
        this.updatePalette();
        this.updateInfo();
        this.saveState();
    }

    initCanvas() {
        const totalWidth = this.gridWidth * this.pixelSize;
        const totalHeight = this.gridHeight * this.pixelSize;

        this.canvas.width = totalWidth;
        this.canvas.height = totalHeight;
        this.overlayCanvas.width = totalWidth;
        this.overlayCanvas.height = totalHeight;

        this.previewCanvas.width = this.gridWidth;
        this.previewCanvas.height = this.gridHeight;

        // Fill with white background
        this.ctx.fillStyle = '#FFFFFF';
        this.ctx.fillRect(0, 0, totalWidth, totalHeight);

        this.drawGrid();
        this.updatePreview();
        this.updateCanvasInfo();
    }

    setupEventListeners() {
        // Tool buttons
        document.querySelectorAll('.tool-btn').forEach(btn => {
            btn.addEventListener('click', (e) => {
                const tool = e.currentTarget.dataset.tool;
                this.setTool(tool);
            });
        });

        // Canvas events
        this.canvas.addEventListener('mousedown', this.onMouseDown.bind(this));
        this.canvas.addEventListener('mousemove', this.onMouseMove.bind(this));
        this.canvas.addEventListener('mouseup', this.onMouseUp.bind(this));
        this.canvas.addEventListener('mouseleave', this.onMouseUp.bind(this));
        this.canvas.addEventListener('contextmenu', (e) => e.preventDefault());

        // Color picker
        const colorPicker = document.getElementById('colorPicker');
        const hexInput = document.getElementById('hexInput');
        const primaryColorSwatch = document.getElementById('primaryColor');
        const secondaryColorSwatch = document.getElementById('secondaryColor');

        colorPicker.addEventListener('input', (e) => {
            this.primaryColor = e.target.value;
            this.updateColorDisplay();
        });

        hexInput.addEventListener('input', (e) => {
            let hex = e.target.value.replace(/[^0-9A-Fa-f]/g, '');
            if (hex.length === 6) {
                this.primaryColor = '#' + hex;
                this.updateColorDisplay();
            }
        });

        primaryColorSwatch.addEventListener('click', () => {
            colorPicker.click();
        });

        secondaryColorSwatch.addEventListener('click', () => {
            const temp = this.primaryColor;
            this.primaryColor = this.secondaryColor;
            this.secondaryColor = temp;
            this.updateColorDisplay();
        });

        document.getElementById('swapColors').addEventListener('click', () => {
            const temp = this.primaryColor;
            this.primaryColor = this.secondaryColor;
            this.secondaryColor = temp;
            this.updateColorDisplay();
        });

        // Add to palette
        document.getElementById('addToPalette').addEventListener('click', () => {
            if (!this.palette.includes(this.primaryColor)) {
                this.palette.push(this.primaryColor);
                this.updatePalette();
            }
        });

        // Canvas size
        document.getElementById('resizeCanvas').addEventListener('click', () => {
            const width = parseInt(document.getElementById('canvasWidth').value);
            const height = parseInt(document.getElementById('canvasHeight').value);
            if (width >= 8 && width <= 128 && height >= 8 && height <= 128) {
                this.resizeCanvas(width, height);
            }
        });

        // Clear canvas
        document.getElementById('clearCanvas').addEventListener('click', () => {
            if (confirm('Clear the entire canvas?')) {
                this.ctx.fillStyle = '#FFFFFF';
                this.ctx.fillRect(0, 0, this.canvas.width, this.canvas.height);
                this.saveState();
                this.updatePreview();
            }
        });

        // Export PNG
        document.getElementById('exportPNG').addEventListener('click', () => {
            this.exportPNG();
        });

        // Grid toggle
        document.getElementById('toggleGrid').addEventListener('change', (e) => {
            this.showGrid = e.target.checked;
            this.drawGrid();
        });

        // Zoom controls
        document.getElementById('zoomIn').addEventListener('click', () => {
            this.setZoom(this.zoom + 0.25);
        });

        document.getElementById('zoomOut').addEventListener('click', () => {
            this.setZoom(this.zoom - 0.25);
        });

        document.getElementById('zoomReset').addEventListener('click', () => {
            this.setZoom(1);
        });

        // Undo/Redo
        document.getElementById('undo').addEventListener('click', () => {
            this.undo();
        });

        document.getElementById('redo').addEventListener('click', () => {
            this.redo();
        });

        // Keyboard shortcuts
        document.addEventListener('keydown', (e) => {
            if (e.target.tagName === 'INPUT') return;

            const key = e.key.toLowerCase();

            // Tools
            if (key === 'p') this.setTool('pencil');
            else if (key === 'e') this.setTool('eraser');
            else if (key === 'f') this.setTool('fill');
            else if (key === 'i') this.setTool('eyedropper');
            else if (key === 'l') this.setTool('line');
            else if (key === 'r') this.setTool('rectangle');
            else if (key === 'c') this.setTool('circle');
            else if (key === 'x') {
                const temp = this.primaryColor;
                this.primaryColor = this.secondaryColor;
                this.secondaryColor = temp;
                this.updateColorDisplay();
            }

            // Undo/Redo
            if (e.ctrlKey || e.metaKey) {
                if (key === 'z' && !e.shiftKey) {
                    e.preventDefault();
                    this.undo();
                } else if (key === 'y' || (key === 'z' && e.shiftKey)) {
                    e.preventDefault();
                    this.redo();
                }
            }
        });
    }

    setTool(tool) {
        this.currentTool = tool;
        document.querySelectorAll('.tool-btn').forEach(btn => {
            btn.classList.toggle('active', btn.dataset.tool === tool);
        });
        this.updateInfo();
    }

    updateColorDisplay() {
        document.getElementById('colorPicker').value = this.primaryColor;
        document.getElementById('hexInput').value = this.primaryColor.substring(1);
        document.getElementById('primaryColor').style.backgroundColor = this.primaryColor;
        document.getElementById('secondaryColor').style.backgroundColor = this.secondaryColor;
        this.updateInfo();
    }

    updatePalette() {
        const paletteGrid = document.getElementById('paletteGrid');
        paletteGrid.innerHTML = '';

        this.palette.forEach(color => {
            const colorDiv = document.createElement('div');
            colorDiv.className = 'palette-color';
            colorDiv.style.backgroundColor = color;
            colorDiv.title = color;

            colorDiv.addEventListener('click', (e) => {
                if (e.button === 0) {
                    this.primaryColor = color;
                    this.updateColorDisplay();
                }
            });

            colorDiv.addEventListener('contextmenu', (e) => {
                e.preventDefault();
                this.secondaryColor = color;
                this.updateColorDisplay();
            });

            paletteGrid.appendChild(colorDiv);
        });
    }

    onMouseDown(e) {
        this.isDrawing = true;
        this.currentButton = e.button;
        const pos = this.getGridPosition(e);

        if (this.currentTool === 'eyedropper') {
            this.pickColor(pos);
            return;
        }

        if (this.currentTool === 'fill') {
            const color = this.currentButton === 2 ? this.secondaryColor : this.primaryColor;
            this.floodFill(pos.x, pos.y, color);
            this.saveState();
            this.updatePreview();
            return;
        }

        if (['line', 'rectangle', 'circle'].includes(this.currentTool)) {
            this.startPos = pos;
        } else {
            this.drawPixel(pos);
        }
    }

    onMouseMove(e) {
        if (!this.isDrawing) return;

        const pos = this.getGridPosition(e);

        if (['line', 'rectangle', 'circle'].includes(this.currentTool) && this.startPos) {
            this.drawShapePreview(this.startPos, pos);
        } else if (['pencil', 'eraser'].includes(this.currentTool)) {
            this.drawPixel(pos);
        }
    }

    onMouseUp(e) {
        if (!this.isDrawing) return;

        if (['line', 'rectangle', 'circle'].includes(this.currentTool) && this.startPos) {
            const pos = this.getGridPosition(e);
            this.drawShape(this.startPos, pos);
            this.clearOverlay();
            this.startPos = null;
        }

        this.isDrawing = false;
        this.saveState();
        this.updatePreview();
    }

    getGridPosition(e) {
        const rect = this.canvas.getBoundingClientRect();
        const scaleX = this.canvas.width / rect.width;
        const scaleY = this.canvas.height / rect.height;

        const x = Math.floor(((e.clientX - rect.left) * scaleX) / this.pixelSize);
        const y = Math.floor(((e.clientY - rect.top) * scaleY) / this.pixelSize);

        return {
            x: Math.max(0, Math.min(x, this.gridWidth - 1)),
            y: Math.max(0, Math.min(y, this.gridHeight - 1))
        };
    }

    drawPixel(pos) {
        const color = this.currentButton === 2 ? this.secondaryColor : this.primaryColor;
        const fillColor = this.currentTool === 'eraser' ? '#FFFFFF' : color;

        this.ctx.fillStyle = fillColor;
        this.ctx.fillRect(
            pos.x * this.pixelSize,
            pos.y * this.pixelSize,
            this.pixelSize,
            this.pixelSize
        );
    }

    pickColor(pos) {
        const imageData = this.ctx.getImageData(
            pos.x * this.pixelSize,
            pos.y * this.pixelSize,
            1,
            1
        );
        const [r, g, b] = imageData.data;
        const hex = '#' + [r, g, b].map(x => x.toString(16).padStart(2, '0')).join('');

        if (this.currentButton === 2) {
            this.secondaryColor = hex;
        } else {
            this.primaryColor = hex;
        }
        this.updateColorDisplay();
    }

    floodFill(startX, startY, fillColor) {
        const imageData = this.ctx.getImageData(0, 0, this.canvas.width, this.canvas.height);
        const pixels = imageData.data;

        const startPixel = (startY * this.pixelSize * this.canvas.width + startX * this.pixelSize) * 4;
        const startR = pixels[startPixel];
        const startG = pixels[startPixel + 1];
        const startB = pixels[startPixel + 2];

        const fillRGB = this.hexToRgb(fillColor);

        if (startR === fillRGB.r && startG === fillRGB.g && startB === fillRGB.b) {
            return;
        }

        const stack = [[startX, startY]];
        const visited = new Set();

        while (stack.length > 0) {
            const [x, y] = stack.pop();
            const key = `${x},${y}`;

            if (visited.has(key)) continue;
            if (x < 0 || x >= this.gridWidth || y < 0 || y >= this.gridHeight) continue;

            visited.add(key);

            const pixelIndex = (y * this.pixelSize * this.canvas.width + x * this.pixelSize) * 4;
            const r = pixels[pixelIndex];
            const g = pixels[pixelIndex + 1];
            const b = pixels[pixelIndex + 2];

            if (r !== startR || g !== startG || b !== startB) continue;

            this.ctx.fillStyle = fillColor;
            this.ctx.fillRect(x * this.pixelSize, y * this.pixelSize, this.pixelSize, this.pixelSize);

            stack.push([x + 1, y], [x - 1, y], [x, y + 1], [x, y - 1]);
        }
    }

    drawShapePreview(start, end) {
        this.clearOverlay();
        this.overlayCtx.strokeStyle = this.currentButton === 2 ? this.secondaryColor : this.primaryColor;
        this.overlayCtx.lineWidth = 2;
        this.overlayCtx.setLineDash([5, 5]);

        if (this.currentTool === 'line') {
            this.overlayCtx.beginPath();
            this.overlayCtx.moveTo(
                start.x * this.pixelSize + this.pixelSize / 2,
                start.y * this.pixelSize + this.pixelSize / 2
            );
            this.overlayCtx.lineTo(
                end.x * this.pixelSize + this.pixelSize / 2,
                end.y * this.pixelSize + this.pixelSize / 2
            );
            this.overlayCtx.stroke();
        } else if (this.currentTool === 'rectangle') {
            const x = Math.min(start.x, end.x) * this.pixelSize;
            const y = Math.min(start.y, end.y) * this.pixelSize;
            const w = (Math.abs(end.x - start.x) + 1) * this.pixelSize;
            const h = (Math.abs(end.y - start.y) + 1) * this.pixelSize;
            this.overlayCtx.strokeRect(x, y, w, h);
        } else if (this.currentTool === 'circle') {
            const centerX = (start.x + end.x) / 2 * this.pixelSize + this.pixelSize / 2;
            const centerY = (start.y + end.y) / 2 * this.pixelSize + this.pixelSize / 2;
            const radiusX = Math.abs(end.x - start.x) / 2 * this.pixelSize;
            const radiusY = Math.abs(end.y - start.y) / 2 * this.pixelSize;

            this.overlayCtx.beginPath();
            this.overlayCtx.ellipse(centerX, centerY, radiusX, radiusY, 0, 0, 2 * Math.PI);
            this.overlayCtx.stroke();
        }

        this.overlayCtx.setLineDash([]);
    }

    drawShape(start, end) {
        const color = this.currentButton === 2 ? this.secondaryColor : this.primaryColor;

        if (this.currentTool === 'line') {
            this.drawLine(start.x, start.y, end.x, end.y, color);
        } else if (this.currentTool === 'rectangle') {
            this.drawRectangle(start, end, color);
        } else if (this.currentTool === 'circle') {
            this.drawCircle(start, end, color);
        }
    }

    drawLine(x0, y0, x1, y1, color) {
        // Bresenham's line algorithm
        const dx = Math.abs(x1 - x0);
        const dy = Math.abs(y1 - y0);
        const sx = x0 < x1 ? 1 : -1;
        const sy = y0 < y1 ? 1 : -1;
        let err = dx - dy;

        while (true) {
            this.ctx.fillStyle = color;
            this.ctx.fillRect(x0 * this.pixelSize, y0 * this.pixelSize, this.pixelSize, this.pixelSize);

            if (x0 === x1 && y0 === y1) break;

            const e2 = 2 * err;
            if (e2 > -dy) {
                err -= dy;
                x0 += sx;
            }
            if (e2 < dx) {
                err += dx;
                y0 += sy;
            }
        }
    }

    drawRectangle(start, end, color) {
        const minX = Math.min(start.x, end.x);
        const minY = Math.min(start.y, end.y);
        const maxX = Math.max(start.x, end.x);
        const maxY = Math.max(start.y, end.y);

        this.ctx.fillStyle = color;
        for (let x = minX; x <= maxX; x++) {
            for (let y = minY; y <= maxY; y++) {
                this.ctx.fillRect(x * this.pixelSize, y * this.pixelSize, this.pixelSize, this.pixelSize);
            }
        }
    }

    drawCircle(start, end, color) {
        const centerX = Math.round((start.x + end.x) / 2);
        const centerY = Math.round((start.y + end.y) / 2);
        const radiusX = Math.abs(end.x - start.x) / 2;
        const radiusY = Math.abs(end.y - start.y) / 2;

        this.ctx.fillStyle = color;

        for (let x = 0; x < this.gridWidth; x++) {
            for (let y = 0; y < this.gridHeight; y++) {
                const dx = (x - centerX) / radiusX;
                const dy = (y - centerY) / radiusY;
                if (dx * dx + dy * dy <= 1) {
                    this.ctx.fillRect(x * this.pixelSize, y * this.pixelSize, this.pixelSize, this.pixelSize);
                }
            }
        }
    }

    clearOverlay() {
        this.overlayCtx.clearRect(0, 0, this.overlayCanvas.width, this.overlayCanvas.height);
    }

    drawGrid() {
        this.clearOverlay();

        if (!this.showGrid) return;

        this.overlayCtx.strokeStyle = 'rgba(200, 200, 200, 0.3)';
        this.overlayCtx.lineWidth = 1;

        for (let x = 0; x <= this.gridWidth; x++) {
            this.overlayCtx.beginPath();
            this.overlayCtx.moveTo(x * this.pixelSize, 0);
            this.overlayCtx.lineTo(x * this.pixelSize, this.canvas.height);
            this.overlayCtx.stroke();
        }

        for (let y = 0; y <= this.gridHeight; y++) {
            this.overlayCtx.beginPath();
            this.overlayCtx.moveTo(0, y * this.pixelSize);
            this.overlayCtx.lineTo(this.canvas.width, y * this.pixelSize);
            this.overlayCtx.stroke();
        }
    }

    updatePreview() {
        this.previewCtx.clearRect(0, 0, this.previewCanvas.width, this.previewCanvas.height);

        for (let x = 0; x < this.gridWidth; x++) {
            for (let y = 0; y < this.gridHeight; y++) {
                const imageData = this.ctx.getImageData(
                    x * this.pixelSize,
                    y * this.pixelSize,
                    1,
                    1
                );
                const [r, g, b, a] = imageData.data;
                this.previewCtx.fillStyle = `rgba(${r},${g},${b},${a / 255})`;
                this.previewCtx.fillRect(x, y, 1, 1);
            }
        }
    }

    resizeCanvas(newWidth, newHeight) {
        const tempCanvas = document.createElement('canvas');
        const tempCtx = tempCanvas.getContext('2d');
        tempCanvas.width = this.canvas.width;
        tempCanvas.height = this.canvas.height;
        tempCtx.drawImage(this.canvas, 0, 0);

        this.gridWidth = newWidth;
        this.gridHeight = newHeight;
        this.initCanvas();

        const copyWidth = Math.min(tempCanvas.width, this.canvas.width);
        const copyHeight = Math.min(tempCanvas.height, this.canvas.height);
        this.ctx.drawImage(tempCanvas, 0, 0, copyWidth, copyHeight, 0, 0, copyWidth, copyHeight);

        this.saveState();
        this.updatePreview();
    }

    setZoom(newZoom) {
        this.zoom = Math.max(0.25, Math.min(4, newZoom));
        const wrapper = document.getElementById('canvasWrapper');
        wrapper.style.transform = `scale(${this.zoom})`;
        document.getElementById('zoomLevel').textContent = `${Math.round(this.zoom * 100)}%`;
    }

    saveState() {
        const imageData = this.ctx.getImageData(0, 0, this.canvas.width, this.canvas.height);

        if (this.historyStep < this.history.length - 1) {
            this.history = this.history.slice(0, this.historyStep + 1);
        }

        this.history.push(imageData);

        if (this.history.length > this.maxHistory) {
            this.history.shift();
        } else {
            this.historyStep++;
        }
    }

    undo() {
        if (this.historyStep > 0) {
            this.historyStep--;
            this.ctx.putImageData(this.history[this.historyStep], 0, 0);
            this.updatePreview();
        }
    }

    redo() {
        if (this.historyStep < this.history.length - 1) {
            this.historyStep++;
            this.ctx.putImageData(this.history[this.historyStep], 0, 0);
            this.updatePreview();
        }
    }

    async exportPNG() {
        const exportCanvas = document.createElement('canvas');
        const exportCtx = exportCanvas.getContext('2d');
        exportCanvas.width = this.gridWidth;
        exportCanvas.height = this.gridHeight;

        for (let x = 0; x < this.gridWidth; x++) {
            for (let y = 0; y < this.gridHeight; y++) {
                const imageData = this.ctx.getImageData(
                    x * this.pixelSize,
                    y * this.pixelSize,
                    1,
                    1
                );
                const [r, g, b, a] = imageData.data;
                exportCtx.fillStyle = `rgba(${r},${g},${b},${a / 255})`;
                exportCtx.fillRect(x, y, 1, 1);
            }
        }

        const dataURL = exportCanvas.toDataURL('image/png');
        const result = await ipcRenderer.invoke('save-png', dataURL);

        if (result.success) {
            alert(`Image saved successfully to:\n${result.path}`);
        }
    }

    updateInfo() {
        document.getElementById('toolInfo').textContent = this.currentTool.charAt(0).toUpperCase() + this.currentTool.slice(1);
        document.getElementById('colorInfo').textContent = this.primaryColor;
    }

    updateCanvasInfo() {
        document.getElementById('canvasInfo').textContent = `${this.gridWidth} x ${this.gridHeight}`;
        document.getElementById('canvasWidth').value = this.gridWidth;
        document.getElementById('canvasHeight').value = this.gridHeight;
    }

    hexToRgb(hex) {
        const result = /^#?([a-f\d]{2})([a-f\d]{2})([a-f\d]{2})$/i.exec(hex);
        return result ? {
            r: parseInt(result[1], 16),
            g: parseInt(result[2], 16),
            b: parseInt(result[3], 16)
        } : { r: 0, g: 0, b: 0 };
    }
}

// Initialize app when DOM is ready
document.addEventListener('DOMContentLoaded', () => {
    new PixelArtApp();
});
